/* -*-c++-*- Copyright (C) 2018 Advanced Driver Information Technology.
 *
 * This library is open source and may be redistributed and/or modified under
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * OpenSceneGraph Public License for more details.
*/

#include "WLSeat.hpp"

#include <errno.h>
#include <string.h>
#include "WLKeyboard.hpp"
#include "WLPointer.hpp"
#include "WLTouch.hpp"

using namespace WaylandBackend;

const struct wl_seat_listener WLSeat::seatListener =
{
      WLSeat::seatCapabilities
    , WLSeat::seatName
};

WLSeat::WLSeat(struct wl_event_queue *eventQ):
_wlSeat(NULL),
_wlEventQ(eventQ),
_wlInterfaceName(),
_wlID(0),
_WLSeatName(),
_keyboard_dev(NULL),
_pointer_dev(NULL),
_touch_dev(NULL)

{

}


int
WLSeat::init(struct wl_registry* registry, uint32_t name,
             const char* interface, uint32_t version)
{
    int ret = -EINVAL;

    if (!strcmp(interface, "wl_seat"))
    {
        ret = 0;

        _wlInterfaceName.assign(interface);
        _wlID = name;

        if (NULL != _wlEventQ)
        {
            struct wl_registry* wrappedRegistry = (struct wl_registry*)
                                    wl_proxy_create_wrapper((void *)registry);
            if (NULL != wrappedRegistry)
            {
                wl_proxy_set_queue((struct wl_proxy*)wrappedRegistry, _wlEventQ);
                _wlSeat = (struct wl_seat*)
                           wl_registry_bind(wrappedRegistry, name,
                                            &wl_seat_interface, version);
                wl_proxy_wrapper_destroy(wrappedRegistry);
            }
        }
        else
        {
            /*
             * It is not good idea to use default queue. default queue can be
             * dispatched by undesired threads
             */
            _wlSeat = (struct wl_seat*) wl_registry_bind(registry, name,
                    &wl_seat_interface, version);
        }
        if (NULL != _wlSeat)
            wl_seat_add_listener(_wlSeat, &seatListener, (void*)this);
        else
            ret = -ENOMEM;

    }

    return ret;
}

uint32_t
WLSeat::getWLID()
{
    return _wlID;
}

std::string&
WLSeat::getWLName()
{
    return _wlInterfaceName;
}

std::string&
WLSeat::getSeatName()
{
    return _WLSeatName;
}

void
WLSeat::surfaceDestroyNotification(struct wl_surface* wlSurface)
{
    if (NULL != _keyboard_dev)
        _keyboard_dev->clearIfFocusMatches(wlSurface);

    if (NULL != _pointer_dev)
        _pointer_dev->clearIfFocusMatches(wlSurface);

    if (NULL != _touch_dev)
        _touch_dev->clearIfFocusMatches(wlSurface);
}

void
WLSeat::seatCapabilities(void* data, struct wl_seat* seat, uint32_t caps)
{
    WLSeat *pwlSeat = static_cast<WLSeat *>(data);

    if ((caps & WL_SEAT_CAPABILITY_POINTER) && (NULL == pwlSeat->_pointer_dev))
    {
        pwlSeat->_pointer_dev = new WLPointer(pwlSeat->_wlSeat,
                                              pwlSeat->_wlEventQ);
    }
    else if (!(caps & WL_SEAT_CAPABILITY_POINTER) &&
              (NULL != pwlSeat->_pointer_dev))
    {
        delete pwlSeat->_pointer_dev;
        pwlSeat->_pointer_dev = NULL;
    }

    if ((caps & WL_SEAT_CAPABILITY_KEYBOARD) &&
        (NULL == pwlSeat->_keyboard_dev))
    {
        pwlSeat->_keyboard_dev = new WLKeyboard(pwlSeat->_wlSeat,
                                                pwlSeat->_wlEventQ);
    }
    else if (!(caps & WL_SEAT_CAPABILITY_KEYBOARD) &&
              (NULL != pwlSeat->_keyboard_dev))
    {
        delete pwlSeat->_keyboard_dev;
        pwlSeat->_keyboard_dev = NULL;
    }

    if ((caps & WL_SEAT_CAPABILITY_TOUCH) && (NULL == pwlSeat->_touch_dev))
    {
        pwlSeat->_touch_dev = new WLTouch(pwlSeat->_wlSeat,
                                          pwlSeat->_wlEventQ);
    }
    else if (!(caps & WL_SEAT_CAPABILITY_TOUCH) &&
              (NULL != pwlSeat->_touch_dev))
    {
        delete pwlSeat->_touch_dev;
        pwlSeat->_touch_dev = NULL;
    }
}

void
WLSeat::seatName(void* data, struct wl_seat* seat, const char *name)
{
    WLSeat *pwlSeat = static_cast<WLSeat *>(data);
    pwlSeat->_WLSeatName.assign(name);
}


WLSeat::~WLSeat()
{
    if (NULL != _touch_dev)
        delete _touch_dev;

    if (NULL != _pointer_dev)
        delete _pointer_dev;

    if (NULL != _keyboard_dev)
        delete _keyboard_dev;

    if (NULL != _wlSeat)
    {
        wl_seat_destroy(_wlSeat);
    }
}

